<?php
// ============================================================
// BufferService.php
//
// Methods:
//   bulkInsert() -> sirf naye leads insert karega
//                   - lead_id already exist kare -> SKIPPED
//                   - naya lead -> INSERTED
//
//   bulkUpdate() -> sirf existing leads update karega
//                   - lead_id exist nahi kare -> NOT_FOUND
//                   - exist kare -> UPDATED
//
// Response format:
//   { "leadId": "...", "status": "INSERTED|SKIPPED|UPDATED|NOT_FOUND|FAILED", "message": "..." }
//
// DB Tables:
//   Table 1: master_buffer       -> core lead info
//   Table 2: staging_buffer_new  -> all staging/detail columns
// ============================================================

require_once __DIR__ . '/../config/database.php';

class BufferService
{
    private PDO $pdo;

    public function __construct()
    {
        $this->pdo = getConnection();
    }

    // ==========================================================
    // bulkInsert: POST /api/buffer/bulk-insert
    // Sirf naye leads insert karega
    // ==========================================================
    public function bulkInsert(array $requests): array
    {
        $responseList = [];

        foreach ($requests as $req) {
            $leadId = $req['leadId'] ?? null;

            if (!$leadId) {
                $responseList[] = ['leadId' => null, 'status' => 'FAILED', 'message' => 'leadId is required'];
                continue;
            }

            try {
                // Already exist karta hai? -> SKIP
                if ($this->existsByLeadId($leadId)) {
                    $responseList[] = [
                        'leadId'  => $leadId,
                        'status'  => 'EXISTS',
                        'message' => 'Lead already exists, use bulk-update endpoint to update'
                    ];
                    continue;
                }

                $this->pdo->beginTransaction();
                $this->insertMasterBuffer($leadId, $req);
                $this->insertStagingBuffer($leadId, $req);
                $this->pdo->commit();

                $responseList[] = ['leadId' => $leadId, 'status' => 'INSERTED', 'message' => 'Lead inserted successfully'];

            } catch (Exception $e) {
                if ($this->pdo->inTransaction()) $this->pdo->rollBack();
                $responseList[] = ['leadId' => $leadId, 'status' => 'FAILED', 'message' => $e->getMessage()];
            }
        }

        return $responseList;
    }

    // ==========================================================
    // bulkUpdate: POST /api/buffer/bulk-update
    // Sirf existing leads update karega
    // ==========================================================
    public function bulkUpdate(array $requests): array
    {
        $responseList = [];

        foreach ($requests as $req) {
            $leadId = $req['leadId'] ?? null;

            if (!$leadId) {
                $responseList[] = ['leadId' => null, 'status' => 'FAILED', 'message' => 'leadId is required'];
                continue;
            }

            try {
                // Exist nahi karta? -> NOT_FOUND
                if (!$this->existsByLeadId($leadId)) {
                    $responseList[] = [
                        'leadId'  => $leadId,
                        'status'  => 'NOT_FOUND',
                        'message' => 'Lead not found, use bulk-insert endpoint to insert'
                    ];
                    continue;
                }

                $this->pdo->beginTransaction();
                $this->updateMasterBuffer($leadId, $req);
                $this->updateStagingBuffer($leadId, $req);
                $this->pdo->commit();

                $responseList[] = ['leadId' => $leadId, 'status' => 'UPDATED', 'message' => 'Lead updated successfully'];

            } catch (Exception $e) {
                if ($this->pdo->inTransaction()) $this->pdo->rollBack();
                $responseList[] = ['leadId' => $leadId, 'status' => 'FAILED', 'message' => $e->getMessage()];
            }
        }

        return $responseList;
    }

    // ==========================================================
    // INSERT: master_buffer
    // ==========================================================
    private function insertMasterBuffer(string $leadId, array $req): void
    {
        $sql = "
            INSERT INTO master_buffer (
                Timestamp, Date_Time, lead_id, Name_of_Client,
                Mobile, Email_Id, Subjects, Notes,
                IVR_URL, WebSite_Name, Data_Source, Verified_Source,
                actual_time, sheet_name
            ) VALUES (
                :timestamp, :dateTime, :leadId, :nameOfClient,
                :mobile, :emailId, :subjects, :notes,
                :ivrUrl, :webSiteName, :dataSource, :verifiedSource,
                :actualTime, :sheetName
            )
        ";
        $this->pdo->prepare($sql)->execute($this->masterParams($leadId, $req));
    }

    // ==========================================================
    // UPDATE: master_buffer
    // ==========================================================
    private function updateMasterBuffer(string $leadId, array $req): void
    {
        $sql = "
            UPDATE master_buffer SET
                Timestamp       = :timestamp,
                Date_Time       = :dateTime,
                Name_of_Client  = :nameOfClient,
                Mobile          = :mobile,
                Email_Id        = :emailId,
                Subjects        = :subjects,
                Notes           = :notes,
                IVR_URL         = :ivrUrl,
                WebSite_Name    = :webSiteName,
                Data_Source     = :dataSource,
                Verified_Source = :verifiedSource,
                actual_time     = :actualTime,
                sheet_name      = :sheetName
            WHERE lead_id = :leadId
        ";
        $this->pdo->prepare($sql)->execute($this->masterParams($leadId, $req));
    }

    // ==========================================================
    // INSERT: staging_buffer_new
    // ==========================================================
    private function insertStagingBuffer(string $leadId, array $req): void
    {
        $sql = "
            INSERT INTO staging_buffer_new (
                Lead_id, Assign_To_MR_Main, Timestamp_2,
                Year, Month, Week, Intent, Duplicate, Sheet_Name,
                UTM_Campaign_Name, UTM_Adgroup_Name, Enquiry_Status_Last,
                Converted_Amount, Converted_Date, Order_Taken_By,
                status, NBD_CRR, KAPPL_KTAHV, Transcription,
                Lead_Relates_to_which_company,
                Name_of_User, Phone_Number_of_User, Email_of_User,
                Country, Priority, Urgency_YES_NO, Contact_Time,
                Summary_of_Conversation, Lead_Outcome, Lead_Category,
                Preferred_Way_to_Contact, gpt_Extraction_Status,
                Sent_status, Test_Col, Mail_Status,
                Reason_why_assign_Or_Delete
            ) VALUES (
                :leadId, :assignToMrMain, :timestamp2,
                :year, :month, :week, :intent, :duplicate, :sheetName,
                :utmCampaignName, :utmAdgroupName, :enquiryStatusLast,
                :convertedAmount, :convertedDate, :orderTakenBy,
                :status, :nbdCrr, :kapplKtahv, :transcription,
                :leadRelatesToWhichCompany,
                :nameOfUser, :phoneNumberOfUser, :emailOfUser,
                :country, :priority, :urgencyYesNo, :contactTime,
                :summaryOfConversation, :leadOutcome, :leadCategory,
                :preferredWayToContact, :gptExtractionStatus,
                :sentStatus, :testCol, :mailStatus,
                :reasonWhyAssignOrDelete
            )
        ";
        $this->pdo->prepare($sql)->execute($this->stagingParams($leadId, $req));
    }

    // ==========================================================
    // UPDATE: staging_buffer_new
    // ==========================================================
    private function updateStagingBuffer(string $leadId, array $req): void
    {
        $sql = "
            UPDATE staging_buffer_new SET
                Assign_To_MR_Main             = :assignToMrMain,
                Timestamp_2                   = :timestamp2,
                Year                          = :year,
                Month                         = :month,
                Week                          = :week,
                Intent                        = :intent,
                Duplicate                     = :duplicate,
                Sheet_Name                    = :sheetName,
                UTM_Campaign_Name             = :utmCampaignName,
                UTM_Adgroup_Name              = :utmAdgroupName,
                Enquiry_Status_Last           = :enquiryStatusLast,
                Converted_Amount              = :convertedAmount,
                Converted_Date                = :convertedDate,
                Order_Taken_By                = :orderTakenBy,
                status                        = :status,
                NBD_CRR                       = :nbdCrr,
                KAPPL_KTAHV                   = :kapplKtahv,
                Transcription                 = :transcription,
                Lead_Relates_to_which_company = :leadRelatesToWhichCompany,
                Name_of_User                  = :nameOfUser,
                Phone_Number_of_User          = :phoneNumberOfUser,
                Email_of_User                 = :emailOfUser,
                Country                       = :country,
                Priority                      = :priority,
                Urgency_YES_NO                = :urgencyYesNo,
                Contact_Time                  = :contactTime,
                Summary_of_Conversation       = :summaryOfConversation,
                Lead_Outcome                  = :leadOutcome,
                Lead_Category                 = :leadCategory,
                Preferred_Way_to_Contact      = :preferredWayToContact,
                gpt_Extraction_Status         = :gptExtractionStatus,
                Sent_status                   = :sentStatus,
                Test_Col                      = :testCol,
                Mail_Status                   = :mailStatus,
                Reason_why_assign_Or_Delete   = :reasonWhyAssignOrDelete
            WHERE Lead_id = :leadId
        ";
        $this->pdo->prepare($sql)->execute($this->stagingParams($leadId, $req));
    }

    // ==========================================================
    // Helper: master_buffer params
    // ==========================================================
    private function masterParams(string $leadId, array $req): array
    {
        return [
            ':leadId'         => $leadId,
            ':timestamp'      => $req['timestamp']      ?? null,
            ':dateTime'       => $req['dateTime']       ?? null,
            ':nameOfClient'   => $req['nameOfClient']   ?? null,
            ':mobile'         => $req['mobile']         ?? null,
            ':emailId'        => $req['emailId']        ?? null,
            ':subjects'       => $req['subjects']       ?? null,
            ':notes'          => $req['notes']          ?? null,
            ':ivrUrl'         => $req['ivrUrl']         ?? null,
            ':webSiteName'    => $req['webSiteName']    ?? null,
            ':dataSource'     => $req['dataSource']     ?? null,
            ':verifiedSource' => $req['verifiedSource'] ?? null,
            ':actualTime'     => $req['actualTime']     ?? null,
            ':sheetName'      => $req['sheetName']      ?? null,
        ];
    }

    // ==========================================================
    // Helper: staging_buffer_new params
    // ==========================================================
    private function stagingParams(string $leadId, array $req): array
    {
        return [
            ':leadId'                    => $leadId,
            ':assignToMrMain'            => $req['assignToMrMain']            ?? null,
            ':timestamp2'                => $req['timestamp2']                ?? null,
            ':year'                      => $req['year']                      ?? null,
            ':month'                     => $req['month']                     ?? null,
            ':week'                      => $req['week']                      ?? null,
            ':intent'                    => $req['intent']                    ?? null,
            ':duplicate'                 => $req['duplicate']                 ?? null,
            ':sheetName'                 => $req['sheetName']                 ?? null,
            ':utmCampaignName'           => $req['utmCampaignName']           ?? null,
            ':utmAdgroupName'            => $req['utmAdgroupName']            ?? null,
            ':enquiryStatusLast'         => $req['enquiryStatusLast']         ?? null,
            ':convertedAmount'           => $req['convertedAmount']           ?? null,
            ':convertedDate'             => $req['convertedDate']             ?? null,
            ':orderTakenBy'              => $req['orderTakenBy']              ?? null,
            ':status'                    => $req['status']                    ?? null,
            ':nbdCrr'                    => $req['nbdCrr']                    ?? null,
            ':kapplKtahv'                => $req['kapplKtahv']                ?? null,
            ':transcription'             => $req['transcription']             ?? null,
            ':leadRelatesToWhichCompany' => $req['leadRelatesToWhichCompany'] ?? null,
            ':nameOfUser'                => $req['nameOfUser']                ?? null,
            ':phoneNumberOfUser'         => $req['phoneNumberOfUser']         ?? null,
            ':emailOfUser'               => $req['emailOfUser']               ?? null,
            ':country'                   => $req['country']                   ?? null,
            ':priority'                  => $req['priority']                  ?? null,
            ':urgencyYesNo'              => $req['urgencyYesNo']              ?? null,
            ':contactTime'               => $req['contactTime']               ?? null,
            ':summaryOfConversation'     => $req['summaryOfConversation']     ?? null,
            ':leadOutcome'               => $req['leadOutcome']               ?? null,
            ':leadCategory'              => $req['leadCategory']              ?? null,
            ':preferredWayToContact'     => $req['preferredWayToContact']     ?? null,
            ':gptExtractionStatus'       => $req['gptExtractionStatus']       ?? null,
            ':sentStatus'                => $req['sentStatus']                ?? null,
            ':testCol'                   => $req['testCol']                   ?? null,
            ':mailStatus'                => $req['mailStatus']                ?? null,
            ':reasonWhyAssignOrDelete'   => $req['reasonWhyAssignOrDelete']   ?? null,
        ];
    }

    // ==========================================================
    // Helper: Check duplicate in master_buffer
    // ==========================================================
    private function existsByLeadId(string $leadId): bool
    {
        $stmt = $this->pdo->prepare(
            "SELECT COUNT(*) FROM master_buffer WHERE lead_id = :leadId"
        );
        $stmt->execute([':leadId' => $leadId]);
        return (int) $stmt->fetchColumn() > 0;
    }
}
